package activitypub

import (
	"errors"
	"reflect"
	"testing"

	"github.com/google/go-cmp/cmp"
)

func mockCollectionPage(items ...Item) CollectionPage {
	cc := CollectionPage{
		ID:   IRIf("https://example.com", Inbox),
		Type: CollectionPageType,
	}
	if len(items) == 0 {
		cc.Items = make(ItemCollection, 0)
	} else {
		cc.Items = items
		cc.TotalItems = uint(len(items))
	}
	return cc
}

func TestCollectionPageNew(t *testing.T) {
	testValue := ID("test")

	c := CollectionNew(testValue)
	p := CollectionPageNew(c)
	if reflect.DeepEqual(p.Collection, c) {
		t.Errorf("Invalid collection parent '%v'", p.PartOf)
	}
	if p.PartOf != c.GetLink() {
		t.Errorf("Invalid collection '%v'", p.PartOf)
	}
}

func TestCollectionPage_Append(t *testing.T) {
	tests := []struct {
		name    string
		col     CollectionPage
		it      []Item
		wantErr error
	}{
		{
			name: "empty",
			col:  mockCollectionPage(),
			it:   ItemCollection{},
		},
		{
			name: "add one item",
			col:  mockCollectionPage(),
			it: ItemCollection{
				Object{ID: ID("grrr")},
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			defer func() {
				tt.col.Items = tt.col.Items[:0]
				tt.col.TotalItems = 0
			}()
			if err := tt.col.Append(tt.it...); (err != nil) && errors.Is(err, tt.wantErr) {
				t.Errorf("Append() error = %v, wantErr %v", err, tt.wantErr)
			}
			if tt.col.TotalItems != uint(len(tt.it)) {
				t.Errorf("Post Append() %T TotalItems %d different than added count %d", tt.col, tt.col.TotalItems, len(tt.it))
			}
			for _, it := range tt.it {
				if !tt.col.Items.Contains(it) {
					t.Errorf("Post Append() unable to find %s in %T Items %v", tt.col, it.GetLink(), tt.col.Items)
				}
			}
		})
	}
}

func TestCollectionPage_UnmarshalJSON(t *testing.T) {
	p := CollectionPage{}

	dataEmpty := []byte("{}")
	p.UnmarshalJSON(dataEmpty)
	if p.ID != "" {
		t.Errorf("Unmarshaled object should have empty ID, received %q", p.ID)
	}
	if HasTypes(p) {
		t.Errorf("Unmarshaled object should have empty Type, received %q", p.GetType())
	}
	if p.AttributedTo != nil {
		t.Errorf("Unmarshaled object should have empty AttributedTo, received %q", p.AttributedTo)
	}
	if len(p.Name) != 0 {
		t.Errorf("Unmarshaled object should have empty Name, received %q", p.Name)
	}
	if len(p.Summary) != 0 {
		t.Errorf("Unmarshaled object should have empty Summary, received %q", p.Summary)
	}
	if len(p.Content) != 0 {
		t.Errorf("Unmarshaled object should have empty Content, received %q", p.Content)
	}
	if p.TotalItems != 0 {
		t.Errorf("Unmarshaled object should have empty TotalItems, received %d", p.TotalItems)
	}
	if len(p.Items) > 0 {
		t.Errorf("Unmarshaled object should have empty Items, received %v", p.Items)
	}
	if p.URL != nil {
		t.Errorf("Unmarshaled object should have empty URL, received %v", p.URL)
	}
	if !p.Published.IsZero() {
		t.Errorf("Unmarshaled object should have empty Published, received %q", p.Published)
	}
	if !p.StartTime.IsZero() {
		t.Errorf("Unmarshaled object should have empty StartTime, received %q", p.StartTime)
	}
	if !p.Updated.IsZero() {
		t.Errorf("Unmarshaled object should have empty Updated, received %q", p.Updated)
	}
	if p.PartOf != nil {
		t.Errorf("Unmarshaled object should have empty PartOf, received %q", p.PartOf)
	}
	if p.Current != nil {
		t.Errorf("Unmarshaled object should have empty Current, received %q", p.Current)
	}
	if p.First != nil {
		t.Errorf("Unmarshaled object should have empty First, received %q", p.First)
	}
	if p.Last != nil {
		t.Errorf("Unmarshaled object should have empty Last, received %q", p.Last)
	}
	if p.Next != nil {
		t.Errorf("Unmarshaled object should have empty Next, received %q", p.Next)
	}
	if p.Prev != nil {
		t.Errorf("Unmarshaled object should have empty Prev, received %q", p.Prev)
	}
}

func TestCollectionPage_Collection(t *testing.T) {
	id := ID("test")

	c := CollectionNew(id)
	p := CollectionPageNew(c)

	if !reflect.DeepEqual(p.Collection(), p.Items) {
		t.Errorf("Collection items should be equal %v %v", p.Collection(), p.Items)
	}
}

func TestCollectionPage_Count(t *testing.T) {
	id := ID("test")

	c := CollectionNew(id)
	p := CollectionPageNew(c)

	if p.TotalItems != 0 {
		t.Errorf("Object should have empty TotalItems, received %d", p.TotalItems)
	}
	if len(p.Items) > 0 {
		t.Errorf("Empty object should have empty Items, received %v", p.Items)
	}
	if p.Count() != uint(len(p.Items)) {
		t.Errorf("%T.Count() returned %d, expected %d", c, p.Count(), len(p.Items))
	}

	_ = p.Append(IRI("test"))
	if p.TotalItems != 1 {
		t.Errorf("Empty object should have %d TotalItems, received %d", 1, p.TotalItems)
	}
	if p.Count() != uint(len(p.Items)) {
		t.Errorf("%T.Count() returned %d, expected %d", c, p.Count(), len(p.Items))
	}
}

func TestToCollectionPage(t *testing.T) {
	tests := []struct {
		name    string
		it      LinkOrIRI
		want    *CollectionPage
		wantErr error
	}{
		{
			name: "empty",
		},
		{
			name: "Valid CollectionPage",
			it:   CollectionPage{ID: "test", Type: CollectionPageType},
			want: &CollectionPage{ID: "test", Type: CollectionPageType},
		},
		{
			name: "Valid *CollectionPage",
			it:   &CollectionPage{ID: "test", Type: CollectionPageType},
			want: &CollectionPage{ID: "test", Type: CollectionPageType},
		},
		{
			name:    "Valid Collection",
			it:      Collection{ID: "test", Type: CollectionType},
			wantErr: ErrorInvalidType[CollectionPage](Collection{}),
		},
		{
			name:    "Valid *Collection",
			it:      &Collection{ID: "test", Type: CollectionType},
			wantErr: ErrorInvalidType[CollectionPage](new(Collection)),
		},
		{
			name:    "Valid OrderedCollection",
			it:      OrderedCollection{ID: "test", Type: OrderedCollectionType},
			wantErr: ErrorInvalidType[CollectionPage](OrderedCollection{}),
		},
		{
			name:    "Valid *OrderedCollection",
			it:      &OrderedCollection{ID: "test", Type: OrderedCollectionType},
			wantErr: ErrorInvalidType[CollectionPage](new(OrderedCollection)),
		},
		{
			name: "Valid OrderedCollectionPage",
			it:   OrderedCollectionPage{ID: "test", Type: OrderedCollectionPageType},
			want: &CollectionPage{ID: "test", Type: OrderedCollectionPageType},
		},
		{
			name: "Valid *OrderedCollectionPage",
			it:   &OrderedCollectionPage{ID: "test", Type: OrderedCollectionPageType},
			want: &CollectionPage{ID: "test", Type: OrderedCollectionPageType},
		},
		{
			name:    "IRI",
			it:      IRI("https://example.com"),
			wantErr: ErrorInvalidType[CollectionPage](IRI("")),
		},
		{
			name:    "IRIs",
			it:      IRIs{IRI("https://example.com")},
			wantErr: ErrorInvalidType[CollectionPage](IRIs{}),
		},
		{
			name:    "ItemCollection",
			it:      ItemCollection{},
			wantErr: ErrorInvalidType[CollectionPage](ItemCollection{}),
		},
		{
			name:    "Object",
			it:      &Object{ID: "test", Type: ArticleType},
			wantErr: ErrorInvalidType[CollectionPage](&Object{}),
		},
		{
			name:    "Activity",
			it:      &Activity{ID: "test", Type: CreateType},
			wantErr: ErrorInvalidType[CollectionPage](&Activity{}),
		},
		{
			name:    "IntransitiveActivity",
			it:      &IntransitiveActivity{ID: "test", Type: ArriveType},
			wantErr: ErrorInvalidType[CollectionPage](&IntransitiveActivity{}),
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := ToCollectionPage(tt.it)
			if !cmp.Equal(err, tt.wantErr, EquateWeakErrors) {
				t.Errorf("ToCollectionPage() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !cmp.Equal(got, tt.want) {
				t.Errorf("ToCollectionPage() got = %s", cmp.Diff(tt.want, got))
			}
		})
	}
}

func TestCollectionPage_Contains(t *testing.T) {
	t.Skipf("TODO")
}

func TestCollectionPage_GetID(t *testing.T) {
	t.Skipf("TODO")
}

func TestCollectionPage_GetLink(t *testing.T) {
	t.Skipf("TODO")
}

func TestCollectionPage_GetType(t *testing.T) {
	t.Skipf("TODO")
}

func TestCollectionPage_IsCollection(t *testing.T) {
	t.Skipf("TODO")
}

func TestCollectionPage_IsLink(t *testing.T) {
	t.Skipf("TODO")
}

func TestCollectionPage_IsObject(t *testing.T) {
	t.Skipf("TODO")
}

func TestCollectionPage_MarshalJSON(t *testing.T) {
	t.Skipf("TODO")
}

func TestCollectionPage_ItemMatches(t *testing.T) {
	t.Skipf("TODO")
}

func TestCollectionPage_Remove(t *testing.T) {
	tests := []struct {
		name      string
		col       CollectionPage
		toRemove  ItemCollection
		remaining ItemCollection
	}{
		{
			name: "empty",
			col:  mockCollectionPage(),
		},
		{
			name:     "remove one item from empty collection",
			col:      mockCollectionPage(),
			toRemove: ItemCollection{Object{ID: ID("grrr")}},
		},
		{
			name: "remove all from collection",
			col: mockCollectionPage(
				Object{ID: ID("grrr")},
				Activity{ID: ID("one")},
				Actor{ID: ID("jdoe")},
			),
			toRemove: ItemCollection{
				Object{ID: ID("grrr")},
				Activity{ID: ID("one")},
				Actor{ID: ID("jdoe")},
			},
			remaining: ItemCollection{},
		},
		{
			name:      "empty_collection_non_nil_item",
			col:       mockCollectionPage(),
			toRemove:  ItemCollection{&Object{}},
			remaining: ItemCollection{},
		},
		{
			name:      "non_empty_collection_nil_item",
			col:       mockCollectionPage(&Object{ID: "test"}),
			toRemove:  nil,
			remaining: ItemCollection{&Object{ID: "test"}},
		},
		{
			name:     "non_empty_collection_non_contained_item_empty_ID",
			col:      mockCollectionPage(&Object{ID: "test"}),
			toRemove: ItemCollection{&Object{}},
			remaining: ItemCollection{
				&Object{ID: "test"},
			},
		},
		{
			name:     "non_empty_collection_non_contained_item",
			col:      mockCollectionPage(&Object{ID: "test"}),
			toRemove: ItemCollection{&Object{ID: "test123"}},
			remaining: ItemCollection{
				&Object{ID: "test"},
			},
		},
		{
			name:      "non_empty_collection_just_contained_item",
			col:       mockCollectionPage(&Object{ID: "test"}),
			toRemove:  ItemCollection{&Object{ID: "test"}},
			remaining: ItemCollection{},
		},
		{
			name: "non_empty_collection_contained_item_first_pos",
			col: mockCollectionPage(
				&Object{ID: "test"},
				&Object{ID: "test123"},
			),
			toRemove: ItemCollection{&Object{ID: "test"}},
			remaining: ItemCollection{
				&Object{ID: "test123"},
			},
		},
		{
			name: "non_empty_collection_contained_item_not_first_pos",
			col: mockCollectionPage(
				&Object{ID: "test123"},
				&Object{ID: "test"},
				&Object{ID: "test321"},
			),
			toRemove: ItemCollection{&Object{ID: "test"}},
			remaining: ItemCollection{
				&Object{ID: "test123"},
				&Object{ID: "test321"},
			},
		},
		{
			name: "non_empty_collection_contained_item_last_pos",
			col: mockCollectionPage(
				&Object{ID: "test123"},
				&Object{ID: "test"},
			),
			toRemove: ItemCollection{&Object{ID: "test"}},
			remaining: ItemCollection{
				&Object{ID: "test123"},
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			tt.col.Remove(tt.toRemove...)

			if tt.col.TotalItems != uint(len(tt.remaining)) {
				t.Errorf("Post Remove() %T TotalItems %d different than expected %d", tt.col, tt.col.TotalItems, len(tt.remaining))
			}
			for _, it := range tt.remaining {
				if !tt.col.Items.Contains(it) {
					t.Errorf("Post Remove() unable to find %s in %T Items %v", it.GetLink(), tt.col, tt.col.Items)
				}
			}
			for _, it := range tt.toRemove {
				if tt.col.Items.Contains(it) {
					t.Errorf("Post Remove() was still able to find %s in %T Items %v", it.GetLink(), tt.col, tt.col.Items)
				}
			}
		})
	}
}
