// Copyright (C) 2018-2019 John Donoghue <john.donoghue@ieee.org>
// Copyright (C) 2012   Andrius Sutas   <andrius.sutas@gmail.com>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.

#include <octave/oct.h>

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

#ifdef BUILD_SERIAL
#include "serialport_class.h"
#endif

// PKG_ADD: autoload ("__srlp_write__", "serialport.oct");
DEFUN_DLD (__srlp_write__, args, nargout, 
        "-*- texinfo -*-\n\
@deftypefn {Loadable Function} {@var{n} = } __srlp_write__ (@var{serial}, @var{data})\n \
\n\
Write data to a serialport interface.\n \
\n\
@subsubheading Inputs\n \
@var{serial} - instance of @var{octave_serialport} class.@*\
@var{data} - data to be written to the serialport interface. Can be either of String or uint8 type.\n \
\n\
@subsubheading Outputs\n \
Upon successful completion, __srlp_write__() shall return the number of bytes written as the result @var{n}.\n \
@end deftypefn")
{
#ifndef BUILD_SERIAL
  error ("serial: Your system doesn't support the SERIAL interface");
  return octave_value ();
#else
  if (args.length () != 2 || args (0).type_id () != octave_serialport::static_type_id ())
    {
      print_usage ();
      return octave_value (-1);
    }

  octave_serialport *serial = NULL;
  int retval;

  const octave_base_value& rep = args (0).get_rep ();
  serial = &((octave_serialport &)rep);

  if (args (1).is_string ()) // String
    {
      retval = serial->write (args (1).string_value ());
    }
  else if (args (1).is_uint8_type ()) // uint8_t
    {
      NDArray data = args (1).array_value ();
      OCTAVE_LOCAL_BUFFER (uint8_t, buf, (data.numel ()));

      if (buf == NULL)
        {
          error ("__srlp_write__: cannot allocate requested memory");
          return octave_value (-1);  
        }

      for (int i = 0; i < data.numel (); i++)
        buf[i] =  static_cast<uint8_t>(data(i));

      retval = serial->write (buf, data.numel ());
    }
  else
    {
      print_usage ();
      return octave_value (-1);
    }

  return octave_value (retval);
#endif
}
