/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.connect.mirror;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.kafka.clients.admin.Admin;
import org.apache.kafka.clients.admin.ConsumerGroupListing;
import org.apache.kafka.clients.admin.ListConsumerGroupOffsetsSpec;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.config.Config;
import org.apache.kafka.common.config.ConfigDef;
import org.apache.kafka.common.config.ConfigValue;
import org.apache.kafka.common.utils.AppInfoParser;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.connect.connector.Task;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.errors.RetriableException;
import org.apache.kafka.connect.mirror.GroupFilter;
import org.apache.kafka.connect.mirror.MirrorCheckpointConfig;
import org.apache.kafka.connect.mirror.MirrorCheckpointTask;
import org.apache.kafka.connect.mirror.MirrorUtils;
import org.apache.kafka.connect.mirror.Scheduler;
import org.apache.kafka.connect.mirror.SourceAndTarget;
import org.apache.kafka.connect.mirror.TopicFilter;
import org.apache.kafka.connect.source.SourceConnector;
import org.apache.kafka.connect.util.ConnectorUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MirrorCheckpointConnector
extends SourceConnector {
    private static final Logger log = LoggerFactory.getLogger(MirrorCheckpointConnector.class);
    private Scheduler scheduler;
    private MirrorCheckpointConfig config;
    private TopicFilter topicFilter;
    private GroupFilter groupFilter;
    private Admin sourceAdminClient;
    private Admin targetAdminClient;
    private SourceAndTarget sourceAndTarget;
    private Set<String> knownConsumerGroups = null;

    public MirrorCheckpointConnector() {
    }

    MirrorCheckpointConnector(Set<String> knownConsumerGroups, MirrorCheckpointConfig config) {
        this.knownConsumerGroups = knownConsumerGroups;
        this.config = config;
    }

    public void start(Map<String, String> props) {
        this.config = new MirrorCheckpointConfig(props);
        if (!this.config.enabled()) {
            return;
        }
        this.sourceAndTarget = new SourceAndTarget(this.config.sourceClusterAlias(), this.config.targetClusterAlias());
        this.topicFilter = this.config.topicFilter();
        this.groupFilter = this.config.groupFilter();
        this.sourceAdminClient = this.config.forwardingAdmin(this.config.sourceAdminConfig("checkpoint-source-admin"));
        this.targetAdminClient = this.config.forwardingAdmin(this.config.targetAdminConfig("checkpoint-target-admin"));
        this.scheduler = new Scheduler(((Object)((Object)this)).getClass(), this.config.entityLabel(), this.config.adminTimeout());
        this.scheduler.execute(this::createInternalTopics, "creating internal topics");
        this.scheduler.execute(this::loadInitialConsumerGroups, "loading initial consumer groups");
        this.scheduler.scheduleRepeatingDelayed(this::refreshConsumerGroups, this.config.refreshGroupsInterval(), "refreshing consumer groups");
    }

    public void stop() {
        if (!this.config.enabled()) {
            return;
        }
        Utils.closeQuietly((AutoCloseable)this.scheduler, (String)"scheduler");
        Utils.closeQuietly((AutoCloseable)this.topicFilter, (String)"topic filter");
        Utils.closeQuietly((AutoCloseable)this.groupFilter, (String)"group filter");
        Utils.closeQuietly((AutoCloseable)this.sourceAdminClient, (String)"source admin client");
        Utils.closeQuietly((AutoCloseable)this.targetAdminClient, (String)"target admin client");
    }

    public Config validate(Map<String, String> connectorConfigs) {
        List configValues = super.validate(connectorConfigs).configValues();
        MirrorCheckpointConfig.validate(connectorConfigs).forEach((config, errorMsg) -> {
            ConfigValue configValue = configValues.stream().filter(conf -> conf.name().equals(config)).findAny().orElseGet(() -> {
                ConfigValue result = new ConfigValue(config);
                configValues.add(result);
                return result;
            });
            configValue.addErrorMessage(errorMsg);
        });
        return new Config(configValues);
    }

    public Class<? extends Task> taskClass() {
        return MirrorCheckpointTask.class;
    }

    public List<Map<String, String>> taskConfigs(int maxTasks) {
        if (!this.config.enabled() || this.config.emitCheckpointsInterval().isNegative()) {
            return Collections.emptyList();
        }
        if (this.knownConsumerGroups == null) {
            log.debug("Initial consumer loading has not yet completed");
            throw new RetriableException("Timeout while loading consumer groups.");
        }
        if (this.knownConsumerGroups.isEmpty()) {
            return Collections.emptyList();
        }
        int numTasks = Math.min(maxTasks, this.knownConsumerGroups.size());
        List groupsPartitioned = ConnectorUtils.groupPartitions(new ArrayList<String>(this.knownConsumerGroups), (int)numTasks);
        return IntStream.range(0, numTasks).mapToObj(i -> this.config.taskConfigForConsumerGroups((List)groupsPartitioned.get(i), i)).collect(Collectors.toList());
    }

    public ConfigDef config() {
        return MirrorCheckpointConfig.CONNECTOR_CONFIG_DEF;
    }

    public String version() {
        return AppInfoParser.getVersion();
    }

    public boolean alterOffsets(Map<String, String> connectorConfig, Map<Map<String, ?>, Map<String, ?>> offsets) {
        for (Map.Entry<Map<String, ?>, Map<String, ?>> offsetEntry : offsets.entrySet()) {
            Map<String, ?> sourceOffset = offsetEntry.getValue();
            if (sourceOffset == null) continue;
            Map<String, ?> sourcePartition = offsetEntry.getKey();
            if (sourcePartition == null) {
                throw new ConnectException("Source partitions may not be null");
            }
            MirrorUtils.validateSourcePartitionString(sourcePartition, "group");
            MirrorUtils.validateSourcePartitionString(sourcePartition, "topic");
            MirrorUtils.validateSourcePartitionPartition(sourcePartition);
            MirrorUtils.validateSourceOffset(sourcePartition, sourceOffset, true);
        }
        return true;
    }

    private void refreshConsumerGroups() throws InterruptedException, ExecutionException {
        Set<Object> knownConsumerGroups = this.knownConsumerGroups == null ? Collections.emptySet() : this.knownConsumerGroups;
        Set<String> consumerGroups = this.findConsumerGroups();
        HashSet<String> newConsumerGroups = new HashSet<String>(consumerGroups);
        newConsumerGroups.removeAll(knownConsumerGroups);
        HashSet deadConsumerGroups = new HashSet(knownConsumerGroups);
        deadConsumerGroups.removeAll(consumerGroups);
        if (!newConsumerGroups.isEmpty() || !deadConsumerGroups.isEmpty()) {
            log.info("Found {} consumer groups for {}. {} are new. {} were removed. Previously had {}.", new Object[]{consumerGroups.size(), this.sourceAndTarget, newConsumerGroups.size(), deadConsumerGroups.size(), knownConsumerGroups.size()});
            log.debug("Found new consumer groups: {}", newConsumerGroups);
            this.knownConsumerGroups = consumerGroups;
            this.context.requestTaskReconfiguration();
        }
    }

    private void loadInitialConsumerGroups() throws InterruptedException, ExecutionException {
        String connectorName = this.config.connectorName();
        this.knownConsumerGroups = this.findConsumerGroups();
        log.info("Started {} with {} consumer groups.", (Object)connectorName, (Object)this.knownConsumerGroups.size());
        log.debug("Started {} with consumer groups: {}", (Object)connectorName, this.knownConsumerGroups);
    }

    Set<String> findConsumerGroups() throws InterruptedException, ExecutionException {
        List<String> filteredGroups = this.listConsumerGroups().stream().map(ConsumerGroupListing::groupId).filter(this::shouldReplicateByGroupFilter).collect(Collectors.toList());
        HashSet<String> checkpointGroups = new HashSet<String>();
        HashSet<String> irrelevantGroups = new HashSet<String>();
        Map<String, Map<TopicPartition, OffsetAndMetadata>> groupToOffsets = this.listConsumerGroupOffsets(filteredGroups);
        for (String group : filteredGroups) {
            Set consumedTopics = groupToOffsets.get(group).keySet().stream().map(TopicPartition::topic).filter(this::shouldReplicateByTopicFilter).collect(Collectors.toSet());
            if (consumedTopics.isEmpty()) {
                irrelevantGroups.add(group);
                continue;
            }
            checkpointGroups.add(group);
        }
        log.debug("Ignoring the following groups which do not have any offsets for topics that are accepted by the topic filter: {}", irrelevantGroups);
        return checkpointGroups;
    }

    Collection<ConsumerGroupListing> listConsumerGroups() throws InterruptedException, ExecutionException {
        return MirrorUtils.adminCall(() -> (Collection)this.sourceAdminClient.listConsumerGroups().valid().get(), () -> "list consumer groups on " + this.config.sourceClusterAlias() + " cluster");
    }

    private void createInternalTopics() {
        MirrorUtils.createSinglePartitionCompactedTopic(this.config.checkpointsTopic(), this.config.checkpointsTopicReplicationFactor(), this.targetAdminClient);
    }

    Map<String, Map<TopicPartition, OffsetAndMetadata>> listConsumerGroupOffsets(List<String> groups) throws InterruptedException, ExecutionException {
        ListConsumerGroupOffsetsSpec groupOffsetsSpec = new ListConsumerGroupOffsetsSpec();
        Map<String, ListConsumerGroupOffsetsSpec> groupSpecs = groups.stream().collect(Collectors.toMap(group -> group, group -> groupOffsetsSpec));
        return MirrorUtils.adminCall(() -> (Map)this.sourceAdminClient.listConsumerGroupOffsets(groupSpecs).all().get(), () -> String.format("list offsets for consumer groups %s on %s cluster", groups, this.config.sourceClusterAlias()));
    }

    boolean shouldReplicateByGroupFilter(String group) {
        return this.groupFilter.shouldReplicateGroup(group);
    }

    boolean shouldReplicateByTopicFilter(String topic) {
        return this.topicFilter.shouldReplicateTopic(topic);
    }
}

