/* Software floating-point emulation.
   Compute powers of 10 into _BitInt.

   Copyright (C) 2023-2025 Free Software Foundation, Inc.

This file is part of GCC.

GCC is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 3, or (at your option) any later
version.

GCC is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

Under Section 7 of GPL version 3, you are granted additional
permissions described in the GCC Runtime Library Exception, version
3.1, as published by the Free Software Foundation.

You should have received a copy of the GNU General Public License and
a copy of the GCC Runtime Library Exception along with this program;
see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see
<http://www.gnu.org/licenses/>.  */

#include "soft-fp.h"
#include "bitint.h"

#ifdef __BITINT_MAXWIDTH__
# define BIL_VAL(x) ((UBILtype) (x))
# if BIL_TYPE_SIZE == 64
#  define BIL_PAIR(x, y) ((BIL_VAL (x) << 32) | BIL_VAL (y))
#  define BIL_OFF(x, y) (x)
# elif BIL_TYPE_SIZE == 32
#  if __LIBGCC_BITINT_ORDER__ == __ORDER_BIG_ENDIAN__
#   define BIL_PAIR(x, y) BIL_VAL (x), BIL_VAL (y)
#  else
#   define BIL_PAIR(x, y) BIL_VAL (y), BIL_VAL (x)
#  endif
#  define BIL_OFF(x, y) (y)
# else
#  error Unsupported _BitInt limb size
# endif
#if __LIBGCC_BITINT_ORDER__ == __ORDER_BIG_ENDIAN__
# define BIL_SET2(a, b) a, b
# define BIL_SET3(a, b, c) a, b, c
# define BIL_SET4(a, b, c, d) a, b, c, d
# define BIL_SET5(a, b, c, d, e) a, b, c, d, e
# define BIL_SET6(a, b, c, d, e, f) a, b, c, d, e, f
# define BIL_SET7(a, b, c, d, e, f, g) a, b, c, d, e, f, g
# define BIL_SET8(a, b, c, d, e, f, g, h) a, b, c, d, e, f, g, h
# define BIL_SET9(a, b, c, d, e, f, g, h, i) a, b, c, d, e, f, g, h, i
# define BIL_SET10(a, b, c, d, e, f, g, h, i, j) a, b, c, d, e, f, g, h, i, j
# define BIL_SET11(a, b, c, d, e, f, g, h, i, j, k) \
  a, b, c, d, e, f, g, h, i, j, k
# define BIL_SET12(a, b, c, d, e, f, g, h, i, j, k, l) \
  a, b, c, d, e, f, g, h, i, j, k, l
# define BIL_SET13(a, b, c, d, e, f, g, h, i, j, k, l, m) \
  a, b, c, d, e, f, g, h, i, j, k, l, m
# define BIL_SET14(a, b, c, d, e, f, g, h, i, j, k, l, m, n) \
  a, b, c, d, e, f, g, h, i, j, k, l, m, n
# define BIL_SET15(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) \
  a, b, c, d, e, f, g, h, i, j, k, l, m, n, o
#else
# define BIL_SET2(a, b) b, a
# define BIL_SET3(a, b, c) c, b, a
# define BIL_SET4(a, b, c, d) d, c, b, a
# define BIL_SET5(a, b, c, d, e) e, d, c, b, a
# define BIL_SET6(a, b, c, d, e, f) f, e, d, c, b, a
# define BIL_SET7(a, b, c, d, e, f, g) g, f, e, d, c, b, a
# define BIL_SET8(a, b, c, d, e, f, g, h) h, g, f, e, d, c, b, a
# define BIL_SET9(a, b, c, d, e, f, g, h, i) i, h, g, f, e, d, c, b, a
# define BIL_SET10(a, b, c, d, e, f, g, h, i, j) j, i, h, g, f, e, d, c, b, a
# define BIL_SET11(a, b, c, d, e, f, g, h, i, j, k) \
  k, j, i, h, g, f, e, d, c, b, a
# define BIL_SET12(a, b, c, d, e, f, g, h, i, j, k, l) \
  l, k, j, i, h, g, f, e, d, c, b, a
# define BIL_SET13(a, b, c, d, e, f, g, h, i, j, k, l, m) \
  m, l, k, j, i, h, g, f, e, d, c, b, a
# define BIL_SET14(a, b, c, d, e, f, g, h, i, j, k, l, m, n) \
  n, m, l, k, j, i, h, g, f, e, d, c, b, a
# define BIL_SET15(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o) \
  o, n, m, l, k, j, i, h, g, f, e, d, c, b, a
#endif

#include "bitintpow10.h"

/* Set r (_BitInt limbs with rprec bits) to pow10 (n),
   where n is in [0, 6111].  Returns number of least significant
   limbs with just 0s in it.  */

USItype
__bid_pow10bitint (UBILtype *r, SItype rprec, USItype n)
{
  USItype rn = ((USItype) rprec + BIL_TYPE_SIZE - 1) / BIL_TYPE_SIZE;
  if (n <= 256)
    {
      /* No need to multiply anything, just copy it from pow10_limbs
	 array.  */
      USItype low_zeros = (n / 64) * (64 / BIL_TYPE_SIZE);
      UBILtype *p = &pow10_limbs[pow10_offs[n]];
      USItype cnt = pow10_offs[n + 1] - pow10_offs[n];
      if (low_zeros)
	__builtin_memset (r + BITINT_END (rn - low_zeros, 0), '\0',
			  low_zeros * sizeof (UBILtype));
      __builtin_memcpy (r + BITINT_END (rn - low_zeros - cnt, low_zeros),
			p, cnt * sizeof (UBILtype));
      if (rn > low_zeros + cnt)
	__builtin_memset (r + BITINT_END (0, low_zeros + cnt), '\0',
			  (rn - low_zeros - cnt) * sizeof (UBILtype));
      return low_zeros;
    }
  else
    {
      USItype m = n / 256;
      n &= 255;
      USItype low_zeros = ((n / 64) + (m * 4)) * (64 / BIL_TYPE_SIZE);
      UBILtype *pm = &pow10_limbs[pow10_offs[m + 255]];
      USItype cntm = pow10_offs[m + 256] - pow10_offs[m + 255];
      UBILtype *pn = &pow10_limbs[pow10_offs[n]];
      USItype cntn = pow10_offs[n + 1] - pow10_offs[n];
      if (low_zeros)
	__builtin_memset (r + BITINT_END (rn - low_zeros, 0), '\0',
			  low_zeros * sizeof (UBILtype));
      __mulbitint3 (r + BITINT_END (0, low_zeros),
		    rprec - low_zeros * BIL_TYPE_SIZE,
		    pm, cntm * BIL_TYPE_SIZE, pn, cntn * BIL_TYPE_SIZE);
      return low_zeros;
    }
}
#endif

#ifndef ENABLE_DECIMAL_BID_FORMAT
/* DPD encoded 10-bit number to decimal table.  */
const unsigned short __dpd_d2bbitint[1024] = {
  0, 1, 2, 3, 4, 5, 6, 7,
  8, 9, 80, 81, 800, 801, 880, 881,
  10, 11, 12, 13, 14, 15, 16, 17,
  18, 19, 90, 91, 810, 811, 890, 891,
  20, 21, 22, 23, 24, 25, 26, 27,
  28, 29, 82, 83, 820, 821, 808, 809,
  30, 31, 32, 33, 34, 35, 36, 37,
  38, 39, 92, 93, 830, 831, 818, 819,
  40, 41, 42, 43, 44, 45, 46, 47,
  48, 49, 84, 85, 840, 841, 88, 89,
  50, 51, 52, 53, 54, 55, 56, 57,
  58, 59, 94, 95, 850, 851, 98, 99,
  60, 61, 62, 63, 64, 65, 66, 67,
  68, 69, 86, 87, 860, 861, 888, 889,
  70, 71, 72, 73, 74, 75, 76, 77,
  78, 79, 96, 97, 870, 871, 898, 899,
  100, 101, 102, 103, 104, 105, 106, 107,
  108, 109, 180, 181, 900, 901, 980, 981,
  110, 111, 112, 113, 114, 115, 116, 117,
  118, 119, 190, 191, 910, 911, 990, 991,
  120, 121, 122, 123, 124, 125, 126, 127,
  128, 129, 182, 183, 920, 921, 908, 909,
  130, 131, 132, 133, 134, 135, 136, 137,
  138, 139, 192, 193, 930, 931, 918, 919,
  140, 141, 142, 143, 144, 145, 146, 147,
  148, 149, 184, 185, 940, 941, 188, 189,
  150, 151, 152, 153, 154, 155, 156, 157,
  158, 159, 194, 195, 950, 951, 198, 199,
  160, 161, 162, 163, 164, 165, 166, 167,
  168, 169, 186, 187, 960, 961, 988, 989,
  170, 171, 172, 173, 174, 175, 176, 177,
  178, 179, 196, 197, 970, 971, 998, 999,
  200, 201, 202, 203, 204, 205, 206, 207,
  208, 209, 280, 281, 802, 803, 882, 883,
  210, 211, 212, 213, 214, 215, 216, 217,
  218, 219, 290, 291, 812, 813, 892, 893,
  220, 221, 222, 223, 224, 225, 226, 227,
  228, 229, 282, 283, 822, 823, 828, 829,
  230, 231, 232, 233, 234, 235, 236, 237,
  238, 239, 292, 293, 832, 833, 838, 839,
  240, 241, 242, 243, 244, 245, 246, 247,
  248, 249, 284, 285, 842, 843, 288, 289,
  250, 251, 252, 253, 254, 255, 256, 257,
  258, 259, 294, 295, 852, 853, 298, 299,
  260, 261, 262, 263, 264, 265, 266, 267,
  268, 269, 286, 287, 862, 863, 888, 889,
  270, 271, 272, 273, 274, 275, 276, 277,
  278, 279, 296, 297, 872, 873, 898, 899,
  300, 301, 302, 303, 304, 305, 306, 307,
  308, 309, 380, 381, 902, 903, 982, 983,
  310, 311, 312, 313, 314, 315, 316, 317,
  318, 319, 390, 391, 912, 913, 992, 993,
  320, 321, 322, 323, 324, 325, 326, 327,
  328, 329, 382, 383, 922, 923, 928, 929,
  330, 331, 332, 333, 334, 335, 336, 337,
  338, 339, 392, 393, 932, 933, 938, 939,
  340, 341, 342, 343, 344, 345, 346, 347,
  348, 349, 384, 385, 942, 943, 388, 389,
  350, 351, 352, 353, 354, 355, 356, 357,
  358, 359, 394, 395, 952, 953, 398, 399,
  360, 361, 362, 363, 364, 365, 366, 367,
  368, 369, 386, 387, 962, 963, 988, 989,
  370, 371, 372, 373, 374, 375, 376, 377,
  378, 379, 396, 397, 972, 973, 998, 999,
  400, 401, 402, 403, 404, 405, 406, 407,
  408, 409, 480, 481, 804, 805, 884, 885,
  410, 411, 412, 413, 414, 415, 416, 417,
  418, 419, 490, 491, 814, 815, 894, 895,
  420, 421, 422, 423, 424, 425, 426, 427,
  428, 429, 482, 483, 824, 825, 848, 849,
  430, 431, 432, 433, 434, 435, 436, 437,
  438, 439, 492, 493, 834, 835, 858, 859,
  440, 441, 442, 443, 444, 445, 446, 447,
  448, 449, 484, 485, 844, 845, 488, 489,
  450, 451, 452, 453, 454, 455, 456, 457,
  458, 459, 494, 495, 854, 855, 498, 499,
  460, 461, 462, 463, 464, 465, 466, 467,
  468, 469, 486, 487, 864, 865, 888, 889,
  470, 471, 472, 473, 474, 475, 476, 477,
  478, 479, 496, 497, 874, 875, 898, 899,
  500, 501, 502, 503, 504, 505, 506, 507,
  508, 509, 580, 581, 904, 905, 984, 985,
  510, 511, 512, 513, 514, 515, 516, 517,
  518, 519, 590, 591, 914, 915, 994, 995,
  520, 521, 522, 523, 524, 525, 526, 527,
  528, 529, 582, 583, 924, 925, 948, 949,
  530, 531, 532, 533, 534, 535, 536, 537,
  538, 539, 592, 593, 934, 935, 958, 959,
  540, 541, 542, 543, 544, 545, 546, 547,
  548, 549, 584, 585, 944, 945, 588, 589,
  550, 551, 552, 553, 554, 555, 556, 557,
  558, 559, 594, 595, 954, 955, 598, 599,
  560, 561, 562, 563, 564, 565, 566, 567,
  568, 569, 586, 587, 964, 965, 988, 989,
  570, 571, 572, 573, 574, 575, 576, 577,
  578, 579, 596, 597, 974, 975, 998, 999,
  600, 601, 602, 603, 604, 605, 606, 607,
  608, 609, 680, 681, 806, 807, 886, 887,
  610, 611, 612, 613, 614, 615, 616, 617,
  618, 619, 690, 691, 816, 817, 896, 897,
  620, 621, 622, 623, 624, 625, 626, 627,
  628, 629, 682, 683, 826, 827, 868, 869,
  630, 631, 632, 633, 634, 635, 636, 637,
  638, 639, 692, 693, 836, 837, 878, 879,
  640, 641, 642, 643, 644, 645, 646, 647,
  648, 649, 684, 685, 846, 847, 688, 689,
  650, 651, 652, 653, 654, 655, 656, 657,
  658, 659, 694, 695, 856, 857, 698, 699,
  660, 661, 662, 663, 664, 665, 666, 667,
  668, 669, 686, 687, 866, 867, 888, 889,
  670, 671, 672, 673, 674, 675, 676, 677,
  678, 679, 696, 697, 876, 877, 898, 899,
  700, 701, 702, 703, 704, 705, 706, 707,
  708, 709, 780, 781, 906, 907, 986, 987,
  710, 711, 712, 713, 714, 715, 716, 717,
  718, 719, 790, 791, 916, 917, 996, 997,
  720, 721, 722, 723, 724, 725, 726, 727,
  728, 729, 782, 783, 926, 927, 968, 969,
  730, 731, 732, 733, 734, 735, 736, 737,
  738, 739, 792, 793, 936, 937, 978, 979,
  740, 741, 742, 743, 744, 745, 746, 747,
  748, 749, 784, 785, 946, 947, 788, 789,
  750, 751, 752, 753, 754, 755, 756, 757,
  758, 759, 794, 795, 956, 957, 798, 799,
  760, 761, 762, 763, 764, 765, 766, 767,
  768, 769, 786, 787, 966, 967, 988, 989,
  770, 771, 772, 773, 774, 775, 776, 777,
  778, 779, 796, 797, 976, 977, 998, 999
};

/* Decimal to DPD encoded 10-bit number table.  */
const unsigned short __dpd_b2dbitint[1000] = {
  0x000, 0x001, 0x002, 0x003, 0x004,
  0x005, 0x006, 0x007, 0x008, 0x009,
  0x010, 0x011, 0x012, 0x013, 0x014,
  0x015, 0x016, 0x017, 0x018, 0x019,
  0x020, 0x021, 0x022, 0x023, 0x024,
  0x025, 0x026, 0x027, 0x028, 0x029,
  0x030, 0x031, 0x032, 0x033, 0x034,
  0x035, 0x036, 0x037, 0x038, 0x039,
  0x040, 0x041, 0x042, 0x043, 0x044,
  0x045, 0x046, 0x047, 0x048, 0x049,
  0x050, 0x051, 0x052, 0x053, 0x054,
  0x055, 0x056, 0x057, 0x058, 0x059,
  0x060, 0x061, 0x062, 0x063, 0x064,
  0x065, 0x066, 0x067, 0x068, 0x069,
  0x070, 0x071, 0x072, 0x073, 0x074,
  0x075, 0x076, 0x077, 0x078, 0x079,
  0x00a, 0x00b, 0x02a, 0x02b, 0x04a,
  0x04b, 0x06a, 0x06b, 0x04e, 0x04f,
  0x01a, 0x01b, 0x03a, 0x03b, 0x05a,
  0x05b, 0x07a, 0x07b, 0x05e, 0x05f,
  0x080, 0x081, 0x082, 0x083, 0x084,
  0x085, 0x086, 0x087, 0x088, 0x089,
  0x090, 0x091, 0x092, 0x093, 0x094,
  0x095, 0x096, 0x097, 0x098, 0x099,
  0x0a0, 0x0a1, 0x0a2, 0x0a3, 0x0a4,
  0x0a5, 0x0a6, 0x0a7, 0x0a8, 0x0a9,
  0x0b0, 0x0b1, 0x0b2, 0x0b3, 0x0b4,
  0x0b5, 0x0b6, 0x0b7, 0x0b8, 0x0b9,
  0x0c0, 0x0c1, 0x0c2, 0x0c3, 0x0c4,
  0x0c5, 0x0c6, 0x0c7, 0x0c8, 0x0c9,
  0x0d0, 0x0d1, 0x0d2, 0x0d3, 0x0d4,
  0x0d5, 0x0d6, 0x0d7, 0x0d8, 0x0d9,
  0x0e0, 0x0e1, 0x0e2, 0x0e3, 0x0e4,
  0x0e5, 0x0e6, 0x0e7, 0x0e8, 0x0e9,
  0x0f0, 0x0f1, 0x0f2, 0x0f3, 0x0f4,
  0x0f5, 0x0f6, 0x0f7, 0x0f8, 0x0f9,
  0x08a, 0x08b, 0x0aa, 0x0ab, 0x0ca,
  0x0cb, 0x0ea, 0x0eb, 0x0ce, 0x0cf,
  0x09a, 0x09b, 0x0ba, 0x0bb, 0x0da,
  0x0db, 0x0fa, 0x0fb, 0x0de, 0x0df,
  0x100, 0x101, 0x102, 0x103, 0x104,
  0x105, 0x106, 0x107, 0x108, 0x109,
  0x110, 0x111, 0x112, 0x113, 0x114,
  0x115, 0x116, 0x117, 0x118, 0x119,
  0x120, 0x121, 0x122, 0x123, 0x124,
  0x125, 0x126, 0x127, 0x128, 0x129,
  0x130, 0x131, 0x132, 0x133, 0x134,
  0x135, 0x136, 0x137, 0x138, 0x139,
  0x140, 0x141, 0x142, 0x143, 0x144,
  0x145, 0x146, 0x147, 0x148, 0x149,
  0x150, 0x151, 0x152, 0x153, 0x154,
  0x155, 0x156, 0x157, 0x158, 0x159,
  0x160, 0x161, 0x162, 0x163, 0x164,
  0x165, 0x166, 0x167, 0x168, 0x169,
  0x170, 0x171, 0x172, 0x173, 0x174,
  0x175, 0x176, 0x177, 0x178, 0x179,
  0x10a, 0x10b, 0x12a, 0x12b, 0x14a,
  0x14b, 0x16a, 0x16b, 0x14e, 0x14f,
  0x11a, 0x11b, 0x13a, 0x13b, 0x15a,
  0x15b, 0x17a, 0x17b, 0x15e, 0x15f,
  0x180, 0x181, 0x182, 0x183, 0x184,
  0x185, 0x186, 0x187, 0x188, 0x189,
  0x190, 0x191, 0x192, 0x193, 0x194,
  0x195, 0x196, 0x197, 0x198, 0x199,
  0x1a0, 0x1a1, 0x1a2, 0x1a3, 0x1a4,
  0x1a5, 0x1a6, 0x1a7, 0x1a8, 0x1a9,
  0x1b0, 0x1b1, 0x1b2, 0x1b3, 0x1b4,
  0x1b5, 0x1b6, 0x1b7, 0x1b8, 0x1b9,
  0x1c0, 0x1c1, 0x1c2, 0x1c3, 0x1c4,
  0x1c5, 0x1c6, 0x1c7, 0x1c8, 0x1c9,
  0x1d0, 0x1d1, 0x1d2, 0x1d3, 0x1d4,
  0x1d5, 0x1d6, 0x1d7, 0x1d8, 0x1d9,
  0x1e0, 0x1e1, 0x1e2, 0x1e3, 0x1e4,
  0x1e5, 0x1e6, 0x1e7, 0x1e8, 0x1e9,
  0x1f0, 0x1f1, 0x1f2, 0x1f3, 0x1f4,
  0x1f5, 0x1f6, 0x1f7, 0x1f8, 0x1f9,
  0x18a, 0x18b, 0x1aa, 0x1ab, 0x1ca,
  0x1cb, 0x1ea, 0x1eb, 0x1ce, 0x1cf,
  0x19a, 0x19b, 0x1ba, 0x1bb, 0x1da,
  0x1db, 0x1fa, 0x1fb, 0x1de, 0x1df,
  0x200, 0x201, 0x202, 0x203, 0x204,
  0x205, 0x206, 0x207, 0x208, 0x209,
  0x210, 0x211, 0x212, 0x213, 0x214,
  0x215, 0x216, 0x217, 0x218, 0x219,
  0x220, 0x221, 0x222, 0x223, 0x224,
  0x225, 0x226, 0x227, 0x228, 0x229,
  0x230, 0x231, 0x232, 0x233, 0x234,
  0x235, 0x236, 0x237, 0x238, 0x239,
  0x240, 0x241, 0x242, 0x243, 0x244,
  0x245, 0x246, 0x247, 0x248, 0x249,
  0x250, 0x251, 0x252, 0x253, 0x254,
  0x255, 0x256, 0x257, 0x258, 0x259,
  0x260, 0x261, 0x262, 0x263, 0x264,
  0x265, 0x266, 0x267, 0x268, 0x269,
  0x270, 0x271, 0x272, 0x273, 0x274,
  0x275, 0x276, 0x277, 0x278, 0x279,
  0x20a, 0x20b, 0x22a, 0x22b, 0x24a,
  0x24b, 0x26a, 0x26b, 0x24e, 0x24f,
  0x21a, 0x21b, 0x23a, 0x23b, 0x25a,
  0x25b, 0x27a, 0x27b, 0x25e, 0x25f,
  0x280, 0x281, 0x282, 0x283, 0x284,
  0x285, 0x286, 0x287, 0x288, 0x289,
  0x290, 0x291, 0x292, 0x293, 0x294,
  0x295, 0x296, 0x297, 0x298, 0x299,
  0x2a0, 0x2a1, 0x2a2, 0x2a3, 0x2a4,
  0x2a5, 0x2a6, 0x2a7, 0x2a8, 0x2a9,
  0x2b0, 0x2b1, 0x2b2, 0x2b3, 0x2b4,
  0x2b5, 0x2b6, 0x2b7, 0x2b8, 0x2b9,
  0x2c0, 0x2c1, 0x2c2, 0x2c3, 0x2c4,
  0x2c5, 0x2c6, 0x2c7, 0x2c8, 0x2c9,
  0x2d0, 0x2d1, 0x2d2, 0x2d3, 0x2d4,
  0x2d5, 0x2d6, 0x2d7, 0x2d8, 0x2d9,
  0x2e0, 0x2e1, 0x2e2, 0x2e3, 0x2e4,
  0x2e5, 0x2e6, 0x2e7, 0x2e8, 0x2e9,
  0x2f0, 0x2f1, 0x2f2, 0x2f3, 0x2f4,
  0x2f5, 0x2f6, 0x2f7, 0x2f8, 0x2f9,
  0x28a, 0x28b, 0x2aa, 0x2ab, 0x2ca,
  0x2cb, 0x2ea, 0x2eb, 0x2ce, 0x2cf,
  0x29a, 0x29b, 0x2ba, 0x2bb, 0x2da,
  0x2db, 0x2fa, 0x2fb, 0x2de, 0x2df,
  0x300, 0x301, 0x302, 0x303, 0x304,
  0x305, 0x306, 0x307, 0x308, 0x309,
  0x310, 0x311, 0x312, 0x313, 0x314,
  0x315, 0x316, 0x317, 0x318, 0x319,
  0x320, 0x321, 0x322, 0x323, 0x324,
  0x325, 0x326, 0x327, 0x328, 0x329,
  0x330, 0x331, 0x332, 0x333, 0x334,
  0x335, 0x336, 0x337, 0x338, 0x339,
  0x340, 0x341, 0x342, 0x343, 0x344,
  0x345, 0x346, 0x347, 0x348, 0x349,
  0x350, 0x351, 0x352, 0x353, 0x354,
  0x355, 0x356, 0x357, 0x358, 0x359,
  0x360, 0x361, 0x362, 0x363, 0x364,
  0x365, 0x366, 0x367, 0x368, 0x369,
  0x370, 0x371, 0x372, 0x373, 0x374,
  0x375, 0x376, 0x377, 0x378, 0x379,
  0x30a, 0x30b, 0x32a, 0x32b, 0x34a,
  0x34b, 0x36a, 0x36b, 0x34e, 0x34f,
  0x31a, 0x31b, 0x33a, 0x33b, 0x35a,
  0x35b, 0x37a, 0x37b, 0x35e, 0x35f,
  0x380, 0x381, 0x382, 0x383, 0x384,
  0x385, 0x386, 0x387, 0x388, 0x389,
  0x390, 0x391, 0x392, 0x393, 0x394,
  0x395, 0x396, 0x397, 0x398, 0x399,
  0x3a0, 0x3a1, 0x3a2, 0x3a3, 0x3a4,
  0x3a5, 0x3a6, 0x3a7, 0x3a8, 0x3a9,
  0x3b0, 0x3b1, 0x3b2, 0x3b3, 0x3b4,
  0x3b5, 0x3b6, 0x3b7, 0x3b8, 0x3b9,
  0x3c0, 0x3c1, 0x3c2, 0x3c3, 0x3c4,
  0x3c5, 0x3c6, 0x3c7, 0x3c8, 0x3c9,
  0x3d0, 0x3d1, 0x3d2, 0x3d3, 0x3d4,
  0x3d5, 0x3d6, 0x3d7, 0x3d8, 0x3d9,
  0x3e0, 0x3e1, 0x3e2, 0x3e3, 0x3e4,
  0x3e5, 0x3e6, 0x3e7, 0x3e8, 0x3e9,
  0x3f0, 0x3f1, 0x3f2, 0x3f3, 0x3f4,
  0x3f5, 0x3f6, 0x3f7, 0x3f8, 0x3f9,
  0x38a, 0x38b, 0x3aa, 0x3ab, 0x3ca,
  0x3cb, 0x3ea, 0x3eb, 0x3ce, 0x3cf,
  0x39a, 0x39b, 0x3ba, 0x3bb, 0x3da,
  0x3db, 0x3fa, 0x3fb, 0x3de, 0x3df,
  0x00c, 0x00d, 0x10c, 0x10d, 0x20c,
  0x20d, 0x30c, 0x30d, 0x02e, 0x02f,
  0x01c, 0x01d, 0x11c, 0x11d, 0x21c,
  0x21d, 0x31c, 0x31d, 0x03e, 0x03f,
  0x02c, 0x02d, 0x12c, 0x12d, 0x22c,
  0x22d, 0x32c, 0x32d, 0x12e, 0x12f,
  0x03c, 0x03d, 0x13c, 0x13d, 0x23c,
  0x23d, 0x33c, 0x33d, 0x13e, 0x13f,
  0x04c, 0x04d, 0x14c, 0x14d, 0x24c,
  0x24d, 0x34c, 0x34d, 0x22e, 0x22f,
  0x05c, 0x05d, 0x15c, 0x15d, 0x25c,
  0x25d, 0x35c, 0x35d, 0x23e, 0x23f,
  0x06c, 0x06d, 0x16c, 0x16d, 0x26c,
  0x26d, 0x36c, 0x36d, 0x32e, 0x32f,
  0x07c, 0x07d, 0x17c, 0x17d, 0x27c,
  0x27d, 0x37c, 0x37d, 0x33e, 0x33f,
  0x00e, 0x00f, 0x10e, 0x10f, 0x20e,
  0x20f, 0x30e, 0x30f, 0x06e, 0x06f,
  0x01e, 0x01f, 0x11e, 0x11f, 0x21e,
  0x21f, 0x31e, 0x31f, 0x07e, 0x07f,
  0x08c, 0x08d, 0x18c, 0x18d, 0x28c,
  0x28d, 0x38c, 0x38d, 0x0ae, 0x0af,
  0x09c, 0x09d, 0x19c, 0x19d, 0x29c,
  0x29d, 0x39c, 0x39d, 0x0be, 0x0bf,
  0x0ac, 0x0ad, 0x1ac, 0x1ad, 0x2ac,
  0x2ad, 0x3ac, 0x3ad, 0x1ae, 0x1af,
  0x0bc, 0x0bd, 0x1bc, 0x1bd, 0x2bc,
  0x2bd, 0x3bc, 0x3bd, 0x1be, 0x1bf,
  0x0cc, 0x0cd, 0x1cc, 0x1cd, 0x2cc,
  0x2cd, 0x3cc, 0x3cd, 0x2ae, 0x2af,
  0x0dc, 0x0dd, 0x1dc, 0x1dd, 0x2dc,
  0x2dd, 0x3dc, 0x3dd, 0x2be, 0x2bf,
  0x0ec, 0x0ed, 0x1ec, 0x1ed, 0x2ec,
  0x2ed, 0x3ec, 0x3ed, 0x3ae, 0x3af,
  0x0fc, 0x0fd, 0x1fc, 0x1fd, 0x2fc,
  0x2fd, 0x3fc, 0x3fd, 0x3be, 0x3bf,
  0x08e, 0x08f, 0x18e, 0x18f, 0x28e,
  0x28f, 0x38e, 0x38f, 0x0ee, 0x0ef,
  0x09e, 0x09f, 0x19e, 0x19f, 0x29e,
  0x29f, 0x39e, 0x39f, 0x0fe, 0x0ff
};
#endif
