% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/src.R
\name{src}
\alias{src}
\title{Display tree of source references}
\usage{
src(x, max_depth = 5L, max_length = 100L, ...)
}
\arguments{
\item{x}{An R object with source references. Can be:
\itemize{
\item A \code{srcref} object
\item A list of \code{srcref} objects
\item A expression vector with attached source references
\item An evaluated closure with attached source references
\item A quoted call with attached source references
}}

\item{max_depth}{Maximum depth to traverse nested structures (default 5)}

\item{max_length}{Maximum number of srcref nodes to display (default 100)}

\item{...}{Additional arguments passed to \code{\link[=tree]{tree()}}}
}
\value{
Returns a structured list containing the source reference
information. Print it to view the formatted tree.
}
\description{
View source reference metadata attached to R objects in a tree structure.
Shows source file information, line/column locations, and lines of source code.
}
\section{Overview}{


Source references are made of two kinds of objects:
\itemize{
\item \code{srcref} objects, which contain information about a specific
location within the source file, such as the line and column numbers.
\item \code{srcfile} objects, which contain metadata about the source file
such as its name, path, and encoding.
}
\subsection{Where and when are source references created?}{

Ultimately the R parser creates source references. The main two entry points
to the parser are:
\itemize{
\item The R function \code{parse()}.
\item The frontend hook \code{ReadConsole}, which powers the console input parser in
the R CLI and in IDEs. This C-level parser can also be accessed from C code
via \code{R_ParseVector()}.
}

In principle, anything that calls \code{parse()} may create source references, but
here are the important direct and indirect callers:
\itemize{
\item \code{source()} and \code{sys.source()} which parse and evaluate code.
\item \code{loadNamespace()} calls \code{sys.source()} when loading a \emph{source} package:
\url{https://github.com/r-devel/r-svn/blob/acd196be/src/library/base/R/namespace.R#L573}.
\item \verb{R CMD install} creates a lazy-load database from a source package.
The first step is to call \code{loadNamespace()}:
\url{https://github.com/r-devel/r-svn/blob/acd196be/src/library/tools/R/makeLazyLoad.R#L32}
}

By default source references are not created but can be enabled by:
\itemize{
\item Passing \code{keep.source = TRUE} explicitly to \code{parse()}, \code{source()},
\code{sys.source()}, or \code{loadNamespace()}.
\item Setting \code{options(keep.source = TRUE)}. This affects the default arguments
of the aforementioned functions, as well as the console input parser.
In interactive sessions, \code{keep.source} is set to \code{TRUE} by default:
\url{https://github.com/r-devel/r-svn/blob/3a4745af/src/library/profile/Common.R#L26}.
\item Setting \code{options(keep.source.pkgs = TRUE)}. This affects loading a package
from source, and installing a package from source.
}
}

\subsection{\code{srcref} objects}{

\code{srcref} objects are compact integer vectors describing a character range
in a source. It records start/end lines and byte/column positions and,
optionally, the parsed-line numbers if \verb{#line} directives were used.

Lengths of 4, 6, or 8 are allowed:
\itemize{
\item 4: basic (first_line, first_byte, last_line, last_byte). Byte positions
are within the line.
\item 6: adds columns in Unicode codepoints (first_col, last_col)
\item 8: adds parsed-line numbers (first_parsed, last_parsed)
}

The "column" information does not represent grapheme clusters, but Unicode
codepoints. The column cursor is incremented at every UTF-8 lead byte and
there is no support for encodings other than UTF-8.

The srcref columns are right-boundary positions, meaning that for an
expression starting at the start of a line, the column will be 1.
\code{wholeSrcref} (see below) on the other hand starts at 0, before the first
character. It might also end 1 character after the last srcref column.

They are attached as attributes (e.g. \code{attr(x, "srcref")} or \code{attr(x, "wholeSrcref")}), possibly wrapped in a list, to the following objects:
\itemize{
\item Expression vectors returned by \code{parse()} (wrapped in a list)
\item Quoted function calls (unwrapped)
\item Quoted \verb{\{} calls (wrapped in a list). This is crucial for debugging: when R
steps through brace lists, the srcref for the current expression is saved to
a global variable (\code{R_Srcref}) so the IDE knows exactly where execution is
paused. See: \url{https://github.com/r-devel/r-svn/blob/fa0b47c5/src/main/eval.c#L2986}.
\item Evaluated closures (unwrapped)
}

They have a \code{srcfile} attribute that points to the source file.

Methods:
\itemize{
\item \code{as.character()}: Retrieves relevant source lines from the \code{srcfile}
reference.
}
\subsection{\code{wholeSrcref} attributes}{

These are \code{srcref} objects stored in the \code{wholeSrcref} attributes of:
\itemize{
\item Expression vectors returned by \code{parse()}, which seems to be the intended
usage.
\item \verb{\{} calls, which seems unintended.
}

For expression vectors, the \code{wholeSrcref} spans from the first position
to the last position and represents the entire document. For braces, they
span from the first position to the location of the closing brace. There is
no way to know the location of the opening brace without reparsing, which
seems odd. It's probably an overlook from \code{xxexprlist()} calling
\code{attachSrcrefs()} in
\url{https://github.com/r-devel/r-svn/blob/52affc16/src/main/gram.y#L1380}. That
function is also called at the end of parsing, where it's intended for the
\code{wholeSrcref} attribute to be attached.

For evaluated closures, the \code{wholeSrcref} attribute on the body has the same
unreliable start positions as \verb{\{} nodes.
}

}

\subsection{\code{srcfile} objects}{

\code{srcfile} objects are environments representing information about a
source file that a source reference points to. They typically refer to
a file on disk and store the filename, working directory, a timestamp,
and encoding information.

While it is possible to create bare \code{srcfile} objects, specialized subclasses
are much more common.
\subsection{\code{srcfile}}{

A bare \code{srcfile} object does not contain any data apart from the file path.
It lazily loads lines from the file on disk, without any caching.

Fields common to all \code{srcfile} objects:
\itemize{
\item \code{filename}: The filename of the source file. If relative, the path is
resolved against \code{wd}.
\item \code{wd}: The working directory (\code{getwd()}) at the time the srcfile was created,
generally at the time of parsing).
\item \code{timestamp}: The timestamp of the source file. Retrieved from \code{filename}
with \code{file.mtime()}.
\item \code{encoding}: The encoding of the source file.
\item \code{Enc}: The encoding of output lines. Used by \code{getSrcLines()}, which
calls \code{iconv()} when \code{Enc} does not match \code{encoding}.
\item \code{parseData} (optional): Parser information saved when \code{keep.source.data} is
set to \code{TRUE}.
}

Implementations:
\itemize{
\item \code{print()} and \code{summary()} to print information about the source file.
\item \code{open()} and \code{close()} to access the underlying file as a connection.
}

Helpers:
\itemize{
\item \code{getSrcLines()}: Retrieves source lines from a \code{srcfile}.
}
}

\subsection{\code{srcfilecopy}}{

A \code{srcfilecopy} stores the actual source lines in memory in \verb{$lines}.
\code{srcfilecopy} is useful when the original file may change or does not
exist, because it preserves the exact text used by the parser.

This type of srcfile is the most common. It's created by:
\itemize{
\item The R-level \code{parse()} function when \code{text} is supplied:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Creates a `"<text>"` non-file `srcfilecopy`
parse(text = "...", keep.source = TRUE)
}\if{html}{\out{</div>}}
\item The console's input parser when \code{getOption("keep.source")} is \code{TRUE}.
\item \code{sys.source()} when \code{keep.source = TRUE}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sys.source(file, keep.source = TRUE)
}\if{html}{\out{</div>}}

The \code{srcfilecopy} object is timestamped with the file's last modification time.
\url{https://github.com/r-devel/r-svn/blob/52affc16/src/library/base/R/source.R#L273-L276}
}

Fields:
\itemize{
\item \code{filename}: The filename of the source file. If \code{isFile} is \code{FALSE},
the field is non meaningful. For instance \code{parse(text = )} sets it to
\code{"<text>"}, and the console input parser sets it to \code{""}.
\item \code{isFile}: A logical indicating whether the source file exists.
\item \code{fixedNewlines}: If \code{TRUE}, \code{lines} is a character vector of lines with
no embedded \verb{\\n} characters. The \code{getSrcLines()} helper regularises \code{lines}
in this way and sets \code{fixedNewlines} to \code{TRUE}.
}

Note that the C-level parser (used directly mainly when parsing console input)
does not call the R-level constructor and only instantiates the \code{filename}
(set to \code{""}) and \code{lines} fields.
}

\subsection{\code{srcfilealias}}{

This object wraps an existing \code{srcfile} object (stored in \code{original}).  It
allows exposing a different \code{filename} while delegating the open/close/get
lines operations to the \code{srcfile} stored in \code{original}.

The typical way aliases are created is via \verb{#line *line* *filename*}
directives where the optional \verb{*filename*} argument is supplied. These
directives remap the srcref and srcfile of parsed code to a different
location, for example from a temporary file or generated file to the original
location on disk.

Created by \code{install.packages()} when installing a \emph{source} package with \code{keep.source.pkgs} set to \code{TRUE} (see
\url{https://github.com/r-devel/r-svn/blob/52affc16/src/library/tools/R/install.R#L545}), but
\href{https://github.com/r-devel/r-svn/blob/52affc16/src/library/tools/R/admin.R#L308}{only when}:
\itemize{
\item \code{Encoding} was supplied in \code{DESCRIPTION}
\item The system locale is not "C" or "POSIX".
}

The source files are converted to the encoding of the system locale, then
collated in a single source file with \verb{#line} directives mapping them to their
original file names (with full paths):
\url{https://github.com/r-devel/r-svn/blob/52affc16/src/library/tools/R/admin.R#L342}.

Note that the \code{filename} of the \code{original} srcfile incorrectly points to the
package path in the install destination.

Fields:
\itemize{
\item \code{filename}: The virtual file name (or full path) of the parsed code.
\item \code{original}: The actual \code{srcfile} the code was parsed from.
}
}

}
}

\seealso{
\itemize{
\item \code{\link[=srcfile]{srcfile()}}: Base documentation for \code{srcref} and \code{srcfile} objects.
\item \code{\link[=getParseData]{getParseData()}}: Parse information stored when \code{keep.source.data} is \code{TRUE}.
\item Source References (R Journal): \url{https://journal.r-project.org/articles/RJ-2010-010/}
}

Other object inspectors: 
\code{\link{ast}()},
\code{\link{ref}()},
\code{\link{sxp}()}
}
\concept{object inspectors}
